;;; povray.el --- major mode for Povray scene files
;;
;; Author: Peter Boettcher <pwb@andrew.cmu.edu>
;; Maintainer: Peter Toneby <woormie@acc.umu.se>
;; Created: 04 March 1994
;; Version: 2.0
;; Keywords: pov, povray
;;
;; LCD Archive Entry:
;; povray|Peter Boettcher|pwb@andrew.cmu.edu|
;; Major mode for Povray scene files|
;; 24-Sep-97|1.3|~/elisp/povray.el|
;;
;; Copyright (C) 1997 Peter W. Boettcher
;;
;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
;;
;;; Commentary:
;;
;; This major mode for GNU Emacs provides support for editing Povray
;; scene files, rendering and viewing them.  It automatically indents
;; blocks, both {} and #if #end.  It also provides context-sensitive
;; keyword completion and font-lock highlighting, as well as the
;; ability to look up those keywords in the povray docu.
;;
;; It should work for either Xemacs or FSF Emacs, versions >= 20;
;; however, only Xemacs can display pictures.
;;
;; To automatically load pov-mode every time Emacs starts up, put the
;; following line into your .emacs file:
;;
;;      (require 'pov-mode)
;;
;; Of course pov-mode has to be somewhere in your load-path for emacs
;; to find it (Use C-h v load-path to see which directories are in the
;; load-path).
;;
;; NOTE: To achieve any sort of reasonable performance, YOU MUST
;;   byte-compile this package.  In emacs, type M-x byte-compile
;;   and then enter the name of this file.
;;
;; You can customize the behaviour of pov-mode and via the
;; customization menu or by simply entering M-x customize-group pov.
;; In many or even most cases, however, it should be completely
;; sufficient to to rely on the default settings.
;;
;; To learn about the basics, just load a pov-file and press C-h m.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Modified by: Peter Boettcher <pwb@andrew.cmu.edu>
;;  5/8/97:
;;    Added font-lock support for Emacs/XEmacs 19
;;    Indent under `#declare Object=' lines
;;    Corrected comment syntax
;;    Got rid of more remnants from postscript mode
;;    General cleanup
;;    Arbitrarily chose version 1.2
;; 5/8/97:  Version 1.21
;;    fontify-insanely was ignored.  fixed.
;;
;; 9/24/97: Version 1.3
;;    Added indentation for Pov 3 syntax (#if #else, etc)
;;    Preliminary context-sensitive keyword completion
;;
;; 1/13/98 by Peter Boettcher <pwb@andrew.cmu.edu>
;;    Explicitly placed package under GPL
;;    Reorganized comment sections and change log to follow GNU standards
;;    Added simple code for jumping to pov documentation (Thanks to
;;         Benjamin Strautin <bis@acpub.duke.edu> for this code)
;;
;; Modified by: Peter Toneby <woormie@acc.umu.se>
;;  22/3/99: Version 1.99beata1
;;    Added support for Pov3.1s new keywords. (not all, I think...)
;;    Removed atmosphere (and atmosphere_*) (stupid me...)
;;
;; Modified by: Peter Toneby <woormie@acc.umu.se>
;;  23/4/99: Version 1.99beata2
;;    Added support for all new keyword, BUT
;;    Added atmosphere (and atmosphere_*) again
;;    Got Pete Boettchers blessing to continue (but with a note
;;      that said that I should have talked to him first, I'm sorry
;;      for not doing that). Pete also said he was willing to let
;;      me continue the maintainance of this file.
;;    I can't get the pov-keyword-help to work, anyone with knowledge
;;      about elisp can send me a fix for it.
;;    The keyword expansion doesn't work for all keywords, I need to add lots of
;;      stuff and read through the docs to get everything correct.
;;
;; Modified by: Alexander Schmolck <aschmolck@gmx.de>
;; 2000-01-31: Version 2beataXXX
;;    Added working keyword lookup in povuser.txt
;;    Added rendering and viewing from within Emacs and with an external viewer
;;    Added customization and made installation simpler
;;    Added a few other minor details
;;    
;; Modified by: Peter Toneby <woormie@acc.umu.se>
;; 2000-05-24: Version 2
;;    Changed the keyword lookup a little, povuser.txt didn't open as
;;       expected when having set the pov-home-dir and pov-help-file
;;       manually.
;;
;; Original Author:     Kevin O. Grover <grover@isri.unlv.edu>
;;        Cre Date:     04 March 1994
;; This file derived from postscript mode by Chris Maio
;;
;;  Please send bug reports/comments/suggestions to me at
;;        woormie@acc.umu.se
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;;  TODO list:
;;     Vector operations (add <0, .5, 1> to every vector in region)
;;     Clean up completion code
;;     TAGS, to jump to #declared objects elsewhere in the code
;;
;;   Feel free to do any of this and send me code :)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;AS TODO:
;; o templates (?)
;; o c-mode like electric parens (?)
;; o keywords! see <(povuser.txt)>
;; o clean up viewing and rendering code
;; o should render or view be decided on filedates? If so, what
;;   image file-name extensions should be checked?
;; o imenu support
;;AS

;;What is cl???
(require 'cl)

(defconst pov-mode-version '2.0		;XXX
  "The povray mode version.")

(defgroup  pov nil
  "*Major mode for editing povray 3.1 scence files <http://www.povray.org>."
  :group 'languages)

(defcustom povray-command "povray"
  "*Command used to invoke the povray."
  :type 'string
  :group 'pov)

(defcustom pov-external-viewer-command "display"
  "*The external viewer to call."
  :type 'string
  :group 'pov)

(defcustom pov-external-view-options "%s"
  "*The options for the viewer; %s is replaced with the name of the rendered image."
  :type 'string
  :group 'pov)

;[FIXME] the test doesn't work as expected with FSF-emacs.
;;allow user to customize external or internal viewer as defaults if she
;;is using Xemacs; for FSF Emacs assume external, since it can't
;;handle pictures anyway
;(if  (and (boundp running-xemacs) running-xemacs)
;    (defcustom pov-default-view-internal t
;      "*Should the pictures be displayed internally by default?"
;      :type 'boolean
;      :group 'pov)
;  (defvar pov-default-view-internal nil))
(defvar pov-default-view-internal nil)

(defcustom pov-run-default "+i%s"
  "*The default options for the Render command (%s is replaced by the filename)."
  :type 'string
  :group 'pov
  )
(defcustom pov-run-test "res120 -Q3 +i%s"
  "*The default options for the Test Render command (%s is replaced by the filename)."
  :type 'string
  :group 'pov
)
(defcustom pov-run-low "res320 +i%s"
    "*The default options for the Test Render command (%s is replaced by the filename)."
  :type 'string
  :group 'pov
)
(defcustom pov-run-mid "res640 +i%s"
  "*The default options for the Medium Res Render command (%s is replaced by the filename)."
  :type 'string
  :group 'pov
)
(defcustom pov-run-high "res800 +i%s"
  "*The default options for the High Res Render command (%s is replaced by the filename)."
  :type 'string
  :group 'pov
)
(defcustom pov-run-highest "res1k +i%s"
  "*The default options for the Higest Res Render command (%s is replaced by the filename)."
  :type 'string
  :group 'pov
)
(defvar pov-external-view
  "External view")
(defvar pov-internal-view
  "Internal view")
(defvar pov-command-alist (list (list "Render"
				      povray-command pov-run-default
				      '()) ;history for the command
				(list "Test quality render"
				      povray-command pov-run-test
				      '())
				(list "Low quality render"
				      povray-command pov-run-low
				      '())
				(list "Medium quality render"
				      povray-command pov-run-highest
				      '())
				(list "High quality render"
				      povray-command pov-run-high
				      '())
				(list pov-external-view
				      pov-external-viewer-command
				      pov-external-view-options
				      '())
				(list pov-internal-view
				      (list pov-internal-view)
				      '()))
  "the commands to run")

(defcustom pov-home-dir "/usr/local/lib/povray31/"
  "*The directory in which the povray files reside."
  :type 'directory
  :group 'pov)

;;XXX change this to arbitrary dir/file?
;;note (peter)
;; that the line here under didn't work here, so I put the
;; concat in the help function.
; (defcustom pov-help-file (concat pov-home-dir "povuser.txt"))
(defcustom pov-help-file "povuser.txt"
  "*The name of the helpfile."
  :type  'file
  :group 'pov)
(defcustom pov-associate-pov-and-inc-with-pov-mode-flag t
  "*If t then files ending with .pov and .inc will automatically start
pov-mode when loaded, unless those file-endings are already in use."
:type 'boolean
:group 'pov)

(defcustom pov-fontify-insanely t
  "*Non-nil means colorize every povray keyword.  This may take a while on lare files.  Maybe disable this on slow systems."
  :type 'boolean
  :group 'pov)


(defcustom pov-indent-level 4
  "*Indentation to be used inside of PoVray blocks or arrays."
  :type 'integer
  :group 'pov)

(defcustom pov-autoindent-endblocks t
  "*When non-nil, automatically reindents when you type break, end, or else."
  :type 'boolean
  :group 'pov
  )

(defcustom pov-indent-under-declare 2
  "*Indentation under a `#declare Object=' line."
  :type 'integer
  :group 'pov)


(defcustom pov-tab-width 8
  "*Tab stop width for PoV mode."
  :type 'integer
  :group 'pov)

(defcustom pov-turn-on-font-lock t
  "*Turn on syntax highlighting automatically"
  :type 'boolean
  :group 'pov)

;;abbrev support
(defvar pov-mode-abbrev-table nil
  "Abbrev table in use in pov-mode buffers.")
(define-abbrev-table 'pov-mode-abbrev-table ())


(when pov-turn-on-font-lock
  (turn-on-font-lock))

(cond (pov-associate-pov-and-inc-with-pov-mode-flag
       (when (not (assoc "\\.pov\\'" auto-mode-alist))
	 (setq auto-mode-alist
	       (append '(("\\.pov\\'" . pov-mode)) auto-mode-alist)))
       (when (not (assoc "\\.inc\\'" auto-mode-alist))
	 (setq auto-mode-alist
	       (append '(("\\.inc\\'" . pov-mode)) auto-mode-alist)))))


;;END AS
(defun pov-make-tabs (stop)
  (and (< stop 132) (cons stop (pov-make-tabs (+ stop pov-tab-width)))))

(defconst pov-tab-stop-list (pov-make-tabs pov-tab-width)
  "Tab stop list for PoV mode")

(defvar pov-mode-map nil
  "Keymap used in PoV mode buffers")

(defvar pov-mode-syntax-table nil
  "PoV mode syntax table")

(defconst pov-comment-start-regexp "//\\|/\\*"
  "Dual comment value for `comment-start-regexp'.")

(defvar pov-comment-syntax-string ". 124b"
  "PoV hack to handle Emacs/XEmacs foo")

(defvar pov-begin-re "\\<#\\(if\\(n?def\\)?\\|case\\|range\\|switch\\|while\\)\\>")
(defvar pov-end-re  "\\<#break\\|#end\\>")

(defvar pov-else-re "\\<#else\\>")

(defvar pov-begin-end-re (concat
                          pov-begin-re
                          "\\|"
                          pov-end-re
                          "\\|"
                          pov-else-re))

;; associate *.pov and *.inc with pov if flag is set and no other
;; modes already have


(defun pov-setup-syntax-table nil
  (if (or (string-match "Lucid" emacs-version)
          (string-match "XEmacs" emacs-version))
      (setq pov-comment-syntax-string ". 1456"))
  (if pov-mode-syntax-table
      ()
    (setq pov-mode-syntax-table (make-syntax-table))
    (modify-syntax-entry ?_ "w" pov-mode-syntax-table)
    (modify-syntax-entry ?# "w" pov-mode-syntax-table)
    (modify-syntax-entry ?/ pov-comment-syntax-string pov-mode-syntax-table)
    (modify-syntax-entry ?* ". 23" pov-mode-syntax-table)
    (modify-syntax-entry ?\n "> b" pov-mode-syntax-table)
    (set-syntax-table pov-mode-syntax-table)))

; Huge regexp for pov-keyword hilighting.  Organized alphabetically.  Try
; to read at your own risk!
(defconst pov-all-keyword-matcher
  '("\\<\\(a\\(a_\\(level\\|threshold\\)\\|bs\\(orption\\)?\\|cosh?\\|d\\(aptive\\|c_bailout\\)\\|gate\\(_turb\\)?\\|ll\\|lpha\\|mbient\\(_light\\)?\\|ngle\\|p\\(erture\\|pend\\)\\|r\\(c_angle\\|ea_light\\|ray\\)\\|sc\\|sinh?\\|ssumed_gamma\\|t\\(an[2h]?\\|mospher\\(e\\|ic_attenuation\\)\\|tenuating\\)\\|verage\\)\\|b\\(ackground\\|ezier_spline\\|l\\(ack_hole\\|ue\\|ur_samples\\)\\|o\\(unded_by\\|x\\(ed\\|_mapping\\)\\|zo\\)\\|r\\(eak\\|ick\\(_size\\)?\\|ightness\\|illiance\\)\\|ump\\(s\\|y[123]?\\|_map\\|_size\\)\\)\\|c\\(a\\(mera\\|se\\|ustics\\)\\|eil\\|h\\(ecker\\|r\\)\\|l\\(ipped_by\\|ock\\(_delta\\)?\\)\\|o\\(lou?r\\(_map\\)?\\|mpo\\(nent\\|site\\)\\|n\\(cat\\|fidence\\|ic_sweep\\|stant\\|trol[01]\\)\\|sh?\\|unt\\)\\|ra\\(ckle\\|nd\\)\\|ubic_wave\\|ylindrical\\(_mapping\\)?\\)\\|d\\(e\\(bug\\|clare\\|f\\(ault\\|ined\\)\\|grees\\|nsity\\(_file\\|_map\\)?\\|nts\\)\\|i\\(ff\\(erence\\|use\\)\\|mension\\(s\\|_size\\)\\|rection\\|s\\(c\\|tance\\(_maximum\\)?\\)\\|v\\)\\|ust\\(_type\\)?\\)\\|e\\(ccentricity\\|lse\\|mi\\(ssion\\|tting\\)\\|nd\\|rror\\(_bound\\)?\\|x\\(tinction\\|p\\(onent\\)?\\)\\)\\|f\\(a\\(de_\\(distance\\|power\\)\\|l\\(loff\\(_angle\\)?\\|se\\)\\)\\|close\\|i\\(l\\(e_exists\\|ter\\)\\|nish\\|sheye\\)\\|l\\(atness\\|ip\\|oor\\)\\|o\\(cal_point\\|g\\(_alt\\|_offset\\|_type\\)?\\|pen\\)\\|requency\\)\\|g\\(if\\|lo\\(bal_settings\\|wing\\)\\|r\\(adient\\|anite\\|ay_threshold\\|een\\)\\)\\|h\\(alo\\|exagon\\|f_gray_16\\|ierarchy\\|ollow\\|ypercomplex\\)\\|i\\(f\\(def\\|f\\|ndef\\)?\\|mage_map\\|n\\(c\\(idence\\|lude\\)\\|t\\(er\\(ior\\|polate\\|section\\|vals\\)\\)?\\|verse\\)\\|or\\|rid\\(_wavelength\\)?\\)\\|jitter\\|l\\(ambda\\|eopard\\|i\\(ght_source\\|near\\(_spline\\|_sweep\\)?\\)\\|o\\(ca\\(l\\|tion\\)\\|g\\|oks_like\\|ok_at\\|w_error_factor\\)\\)\\|m\\(a\\(cro\\|ndel\\|p_type\\|rble\\|terial\\(_map\\)?\\|trix\\|x\\(_intersections\\|_iteration\\|_trace_level\\|_value\\)?\\)\\|e\\(dia\\(_attenuation\\|_interaction\\)?\\|rge\\|tallic\\)\\|in\\(imum_reuse\\)?\\|od\\|ortar\\)\\|n\\(earest_count\\|o\\(rmal\\(_map\\)?\\|_shadow\\)?\\|umber_of_waves\\)\\|o\\(bject\\|ctaves\\|ff\\(set\\)?\\|mega\\|mnimax\\|n\\(ce\\|ion\\)?\\|pen\\|rthographic\\)\\|p\\(a\\(noramic\\|ttern[123]\\)\\|erspective\\|gm\\|h\\(ase\\|ong\\(_size\\)?\\)\\|i\\(gment\\(_map\\)?\\)?\\|lanar\\(_mapping\\)?\\|ng\\|o\\(int_at\\|[tw]\\|ly_wave\\)\\|pm\\|recision\\|wr\\)\\|qu\\(adratic_spline\\|aternion\\|ick_colou?r\\|ilted\\)\\|r\\(a\\(di\\(al\\|ans\\|osity\\|us\\)\\|inbow\\|mp_wave\\|nd\\|nge\\|tio\\)\\|e\\(ad\\|ciprocal\\|cursion_limit\\|d\\|flection\\(_exponent\\)?\\|fraction\\|nder\\|peat\\)\\|gbf?t?\\|ight\\|ipples\\|otate\\|oughness\\)\\|s\\(amples\\|ca\\(le\\|llop_wave\\|ttering\\)\\|eed\\|hadowless\\|in\\(e_wave\\|h\\)?\\|ky\\(_sphere\\)?\\|lice\\|lope_map\\|mooth\\|or\\|p\\(ecular\\|herical\\(_mapping\\)?\\|iral[12]?\\|otlight\\|otted\\)\\|qrt?\\|t\\(atistics\\|r\\(cmp\\|ength\\|len\\|lwr\\|upr\\)?\\|urm\\)\\|ubstr\\|witch\\|ys\\)\\|t\\(anh?\\|est_camera_[1234]\\|exture\\(_map\\)?\\|ga\\|hickness\\|hreshold\\|ightness\\|ile[2s]\\|r\\(ack\\|ans\\(form\\|late\\|mit\\)\\|iangle_wave\\|ue\\)\\|tf\\|urb\\(ulence\\|_depth\\)\\|ype\\)?\\|u\\(ltra_wide_angle\\|n\\(def\\|ion\\)\\|p\\|se_\\(colou?r\\|index\\)\\|_steps\\)?\\|v\\(a\\(l\\|riance\\|xis_rotate\\)\\|cross\\|dot\\|ersion\\|length\\|normalize\\|olume_\\(object\\|rendered\\)\\|ol_with_light\\|rotate\\|_steps\\)?\\|w\\(a\\(r\\(ning\\|p\\)\\|ter_level\\|ves\\)\\|hile\\|idth\\|ood\\|ri\\(nkles\\|te\\)\\)\\|x\\|y\\(es\\)?\\|z\\)\\>" . font-lock-keyword-face))

(defvar pov-font-lock-keywords
  '(
    ("^[ \t]*\\(#declare\\)\\>[ \t]*\\(\\sw+\\)"  2 font-lock-variable-name-face nil t)
    ("\\<#\\(break\\|case\\|de\\(bug\\|clare\\|fault\\)\\|e\\(lse\\|nd\\|rror\\)\\|f\\(open\\|close\\)\\|if\\(n?def\\)?\\|include\\|local\\|macro\\|r\\(ange\\|ender\\|ead\\)\\|s\\(tatistics\\|witch\\)\\|undef\\|version\\|w\\(arning\\|hile\\|rite\\)\\)\\>" . font-lock-function-name-face)
    ("\\<\\(b\\(bezier_spline\\|icubic_patch\\|lob\\|ox\\)\\|c\\(one\\|ub\\(e\\|ic\\(_spline\\)?\\)\\|ylinder\\)\\|disc\\|height_field\\|julia_fractal\\|lathe\\|mesh\\|p\\(lane\\|oly\\(gon\\)?\\|rism\\)\\|qua\\(rt\\|dr\\)ic\\|s\\(mooth_triangle\\|sor\\|phere\\|uperellipsoid\\)\\|t\\(ext\\|orus\\|riangle\\)\\|Rounded\\(Box\\|Cylinder\\)\\)\\>" . font-lock-type-face))
  "Expressions to highlight in PoV mode.")

   
(defun pov-setup-font-lock nil
  "Find out if we're fontifying insanely"
  (if pov-fontify-insanely
      (setq pov-font-lock-keywords (append (list pov-all-keyword-matcher) pov-font-lock-keywords))))

(defun pov-mode nil
  "Major mode for editing PoV files.

   In this mode, TAB and \\[indent-region] attempt to indent code
based on the position of {} pairs and #-type directives.  The variable
pov-indent-level controls the amount of indentation used inside
arrays and begin/end pairs.  The variable pov-indent-under-declare
determines indent level when you have something like this:
#declare foo =
   some_object {

This mode also provides PoVray keyword fontification using font-lock.
Set pov-fontify-insanely to nil to disable (recommended for large
files!).

\\[pov-complete-word] runs pov-complete-word, which attempts to complete the
current word based on point location.
\\[pov-keyword-help] looks up a povray keyword in the povray documentation.
\\[pov-command-query] will render or display the current file.

\\{pov-mode-map}

\\[pov-mode] calls the value of the variable pov-mode-hook  with no args, if that value is non-nil.
"
  (interactive)
  (kill-all-local-variables)
  (use-local-map pov-mode-map)
  (pov-setup-syntax-table)
  (make-local-variable 'comment-start)
  (make-local-variable 'comment-start-skip)
  (make-local-variable 'comment-end)
  (make-local-variable 'comment-multi-line)
  (make-local-variable 'comment-column)
  (make-local-variable 'indent-line-function)
  (make-local-variable 'tab-stop-list)
  (set-syntax-table pov-mode-syntax-table)
  (setq comment-start "// "
        comment-start-skip "/\\*+ *\\|// *"
        comment-end ""
        comment-multi-line nil
        comment-column 60
        indent-line-function 'pov-indent-line
        tab-stop-list pov-tab-stop-list)
  (setq mode-name "PoV")
  (setq major-mode 'pov-mode)
  (pov-setup-font-lock)
  (make-local-variable 'font-lock-keywords)
  (setq font-lock-keywords pov-font-lock-keywords)
  (setq font-lock-defaults '(pov-font-lock-keywords))
  (run-hooks 'pov-mode-hook))

(defun pov-tab ()
  "Command assigned to the TAB key in PoV mode."
  (interactive)
  (if (save-excursion (skip-chars-backward " \t") (bolp))
      (pov-indent-line)
    (save-excursion
      (pov-indent-line))))

(defun pov-indent-line nil
  "Indents a line of PoV code."
  (interactive)
  (beginning-of-line)
  (delete-horizontal-space)
  (if (pov-top-level-p)
      (pov-indent-top-level)
    (if (not (pov-top-level-p))
        (if (pov-in-star-comment-p)
            (indent-to '2)
          (if (and (< (point) (point-max))
                   (or
                    (eq ?\) (char-syntax (char-after (point))))
                    (or
                     (looking-at "\\<#\\(end\\|break\\)\\>")
                     (and (looking-at "\\<#else\\>")
                          (not (pov-in-switch-p 0))))))
              (pov-indent-close)                ; indent close-delimiter
            (pov-indent-in-block))))))  ; indent line after open delimiter
  
(defun pov-newline nil
  "Terminate line and indent next line."
  (interactive)
  (newline)
  (pov-indent-line))

(defun pov-in-star-comment-p nil
  "Return true if in a star comment"
  (let ((state
         (save-excursion
           (parse-partial-sexp (point-min) (point)))))
    (nth 4 state)))

(defun pov-open nil
  (interactive)
  (insert last-command-char))

(defun pov-close nil
  "Inserts and indents a close delimiter."
  (interactive)
  (insert last-command-char)
  (backward-char 1)
  (pov-indent-close)
  (forward-char 1)
  (blink-matching-open))

(defun pov-indent-close nil
  "Internal function to indent a line containing a close delimiter."
  (if (save-excursion (skip-chars-backward " \t") (bolp))
      (let (x (oldpoint (point)))
        (if (looking-at "#end\\|#else\\|#break")
            (progn
              (goto-char (pov-find-begin 0))
              (if (and (looking-at "#else")
                       (pov-in-switch-p 0))
                  (goto-char (pov-find-begin 0))))
          (forward-char) (backward-sexp))       ;XXX
        (if (and (eq 1 (count-lines (point) oldpoint))
                 (> 1 (- oldpoint (point))))
            (goto-char oldpoint)
          (beginning-of-line)
          (skip-chars-forward " \t")
          (setq x (current-column))
          (goto-char oldpoint)
          (delete-horizontal-space)
          (indent-to x)))))

(defun pov-indent-in-block nil
  "Indent a line which does not open or close a block."
  (let ((goal (pov-block-start)))
    (setq goal (save-excursion
                 (goto-char goal)
                 (back-to-indentation)
                 (if (bolp)
                     pov-indent-level
                   (back-to-indentation)
                   (+ (current-column) pov-indent-level))))
    (indent-to goal)))

(defun pov-indent-top-level nil
  (if (save-excursion 
        (forward-line -1)
        (looking-at "\\<#declare[ \t]+[0-9a-zA-Z_]+[ \t]*=[ \t]*$"))
        (indent-to pov-indent-under-declare)))

;;; returns nil if at top-level, or char pos of beginning of current block

(defun pov-block-start nil
  "Returns the character position of the character following the nearest
enclosing `{' or `begin' keyword."
  (save-excursion
    (let (open (skip 0))
      (setq open (condition-case nil
                     (save-excursion
                       (backward-up-list 1)
                       (1+ (point)))
                   (error nil)))
      (pov-find-begin open))))

(defun pov-find-begin (start)
  "Search backwards from point to START for enclosing `begin' and returns the
character number of the character following `begin' or START if not found."
  (save-excursion
    (let ((depth 1) match)
      (while (and (> depth 0)
                  (pov-re-search-backward pov-begin-end-re start t))
        (setq depth (if (looking-at pov-end-re)
                        (if (and (looking-at "#end")
                                 (pov-in-switch-p start))
                            (progn
                              (pov-re-search-backward "\\<#switch\\>" start t)
                              depth)
                          (+ 1 depth))
                      (if (looking-at "\\<#else\\>")
                          (if (pov-in-switch-p start)
                              (1- depth)
                            depth)
                        (1- depth)))))
      (if (not (eq 0 depth))
          start
        (point)))))


(defun pov-in-switch-p (start)
  "Return t if one level under a switch."
  (save-excursion
    (if (looking-at "\\<#end\\>")
          (pov-re-search-backward pov-begin-end-re start t))
    (beginning-of-line)
    (pov-re-search-backward pov-begin-end-re start t)
    (if (looking-at "\\<#else\\>>") (forward-word -1))
    (while (looking-at "\\<#break\\>")
      (progn
        (pov-re-search-backward "\\<#case\\|#range\\>" start t)
        (pov-re-search-backward pov-begin-end-re start t)))
    (pov-re-search-backward pov-begin-end-re start t)
    (looking-at "\\<#switch\\>")))

(defun pov-top-level-p nil
  "Awful test to see whether we are inside some sort of PoVray block."
  (and (condition-case nil
           (not (scan-lists (point) -1 1))
         (error t))
       (not (pov-find-begin nil))))

(defsubst pov-re-search-backward (REGEXP BOUND NOERROR)
  "Like re-search-backward, but skips over matches in comments or strings"
  (set-match-data '(nil nil))
  (while (and
          (re-search-backward REGEXP BOUND NOERROR)
          (pov-skip-backward-comment-or-string)
          (not (set-match-data '(nil nil))))
    ())
  (match-end 0))

(defun pov-autoindent-endblock nil
  "Hack to automatically reindent end, break, and else."
  (interactive)
  (self-insert-command 1)
  (save-excursion
    (forward-word -1)
    (if (looking-at "\\<#else\\|#end\\|#break\\>")
        (pov-indent-line))))

; Taken from verilog-mode.el
(defun pov-skip-backward-comment-or-string ()
 "Return true if in a string or comment"
 (let ((state 
        (save-excursion
          (parse-partial-sexp (point-min) (point)))))
   (cond
    ((nth 3 state)                      ;Inside string
     (search-backward "\"")
     t)
    ((nth 7 state)                      ;Inside // comment
     (search-backward "//")
     t)
    ((nth 4 state)                      ;Inside /* */ comment
     (search-backward "/*")
     t)
    (t
     nil))))

;;; Completions
(defvar pov-completion-str nil)
(defvar pov-completion-all nil)
(defvar pov-completion-pred nil)
;(defvar pov-completion-buffer-to-use nil)
(defvar pov-completion-flag nil)

(defvar pov-top-level-keywords
  '("global_settings" "camera" "light_source"))

(defvar pov-csg-scope-re 
  "\\<inverse\\|union\\|intersection\\|difference\\|merge\\>")

(defvar pov-solid-primitive-keywords
  '("blob" "box" "cone" "cylinder" "julia_fractal" "height_field" "lathe" "prism" "sphere" "superellipsoid" "sor" "text" "torus"))

(defvar pov-blob-keywords
  '("threshold" "cylinder" "sphere" "component" "hierarchy" "sturm"))

(defvar pov-heightfield-keywords
  '("hierarchy" "smooth" "water_level"))

; Julia Fractal

(defvar pov-prism-keywords
  '("linear_sweep" "conic_sweep" "linear_spline" "quadratic_spline" "cubic_spline" "bezier_spline" "sturm"))

(defvar pov-patch-primitive-keywords
  '("bicubic_patch" "disc" "smooth_triangle" "triangle" "polygon" "mesh"))

(defvar pov-bicubic-keywords
  '("type" "flatness" "u_steps" "v_steps"))

(defvar pov-infinite-solid-keywords
  '("plane" "cubic" "poly" "quadric" "quartic"))

(defvar pov-csg-keywords
  '("inverse" "union" "intersection" "difference" "merge"))

(defvar pov-light-source-keywords
  '("color" "spotlight" "point_at" "radius" "falloff" "tightness" "area_light" "adaptive" "jitter" "looks_like" "shadowless" "cylinder" "fade_distance" "fade_power" "media_attenuation" "media_interaction"))

(defvar pov-object-modifier-keywords
  '("clipped_by" "bounded_by" "hollow" "no_shadow"))

(defvar pov-transformation-keywords
  '("rotate" "scale" "translate" "matrix"))

(defvar pov-camera-keywords
  '("perspective" "orthographic" "fisheye" "ultra_wide_angle" "omnimax" "panoramic" "cylinder" "location" "look_at" "right" "up" "direction" "sky" "angle" "blur_samples" "aperture" "focal_point" "normal" "rotate" "translate"))

(defvar pov-texture-keywords
  '("pigment" "normal" "finish" "halo" "texture_map" "material_map" "boxed" "planar" "cylindrical" "spherical"))

(defvar pov-pigment-keywords
  '("color" "boxed" "brick" "checker" "cylindrical" "hexagon" "color_map" "gradient" "pigment_map" "pigment" "planar" "spherical" "image_map" "quick_color"))

(defvar pov-normal-keywords
  '("slope_map" "normal_map" "bump_map" "bump_size" "boxed" "cylindrical" "planar" "spherical"))

(defvar pov-finish-keywords
  '("ambient" "diffuse" "brilliance" "phong" "phong_size" "specular" "roughness" "metallic" "reflection" "reflection_exponent" "refraction" "ior" "caustics" "fade_distance" "fade_power" "irid" "crand"))

;AS: halo is no longer existent in pov 3.1 so we won't need that
;(defvar pov-halo-keywords
;  '("attenuating" "emitting" "glowing" "dust" "constant" "linear" "cubic" "poly" "planar_mapping" "spherical_mapping" "cylindrical_mapping" "box_mapping" "dust_type" "eccentricity" "max_value" "exponent" "samples" "aa_level" "aa_threshold" "jitter" "turbulence" "octaves" "omega" "lambda" "colour_map" "frequency" "phase" "scale" "rotate" "translate"))

(defvar pov-pattern-keywords
  '("agate" "average" "bozo" "granite" "leopard" "marble" "mandel" "onion" "pattern1" "pattern2" "pattern3" "bumpy1" "bumpy2" "bumpy3" "spiral1" "spiral2" "spotted" "wood" "gradient" "crackle" "colour" "checker" "brick" "hexagon" "image_map" "bump_map" "waves" "ripples" "wrinkles" "bumps" "dents" "quilted" ))

(defvar pov-atmosphere-keywords
  '("type" "distance" "scattering" "eccentricity" "samples" "jitter" "aa_threshold" "aa_level" "colour" "color"))

;;
(defvar pov-media-keywords
  '("intervals" "samples" "confidence" "variance" "ratio" "absorption" "emission" "scattering" "density" "color_map" "colour_map" "density_map"))

(defvar pov-interior-keyword
  '("ior" "caustics" "fade_distance" "fade_power"))

(defvar pov-density-keyword
  '("color_map" "colour_map" "density_map" "boxed" "planar" "cylindrical" "spherical"))

(defvar pov-fog-keywords
 '("fog_type" "distance" "color" "colour" "turbulence" "turb_depth" "omega" "lambda" "octaves" "fog_offset" "fog_alt" "up"))

(defvar pov-global-settings-keywords
  '("adc_bailout" "ambient_light" "assumed_gamma" "hf_gray_16" "irid_wavelength" "max_intersections" "max_trace_level" "number_of_waves" "radiosity"))

(defvar pov-radiosity-keywords
  '("brightness" "count" "distance_maximum" "error_bound" "gray_threshold" "low_error_factor" "minimum_reuse" "nearest_count" "recursion_limit"))

(defvar pov-object-keywords
 '("texture" "pigment" "finish" "halo" "normal"))

;;AS
(defvar pov-keyword-completion-alist
  (mapcar (function
	   (lambda (item) (list item item)))
	   (append pov-atmosphere-keywords pov-media-keywords
		   pov-bicubic-keywords pov-normal-keywords
		   pov-blob-keywords pov-object-keywords
		   pov-camera-keywords pov-pattern-keywords
		   pov-csg-keywords pov-pigment-keywords
		   pov-density-keyword pov-prism-keywords
		   pov-finish-keywords pov-radiosity-keywords
		   pov-fog-keywords pov-texture-keywords
		   pov-heightfield-keywords))) ; pov-halo-keywords 

(defun pov-string-diff (str1 str2)
  "Return index of first letter where STR1 and STR2 differs."
  (catch 'done
    (let ((diff 0))
      (while t
        (if (or (> (1+ diff) (length str1))
                (> (1+ diff) (length str2)))
            (throw 'done diff))
        (or (equal (aref str1 diff) (aref str2 diff))
	    (throw 'done diff))
        (setq diff (1+ diff))))))

(defun pov-get-scope nil
  "Return the scope of the POV source at point"
  (interactive) 
  (save-excursion
    (if (not (pov-top-level-p))
        (progn
        
    (backward-up-list 1)
    (forward-word -1)
    (cond
     ((looking-at "camera")
      (setq pov-completion-list pov-camera-keywords))
     ((looking-at "texture")
      (setq pov-completion-list (append pov-texture-keywords pov-pattern-keywords)))
     ((looking-at "pigment")
      (setq pov-completion-list (append pov-pigment-keywords pov-pattern-keywords)))
     ((looking-at "normal")
      (setq pov-completion-list (append pov-normal-keywords pov-pattern-keywords)))
     ((looking-at "finish")
      (setq pov-completion-list pov-finish-keywords))
;     ((looking-at "halo")
;      (setq pov-completion-list pov-halo-keywords))
     ((looking-at "blob")
      (setq pov-completion-list pov-blob-keywords))
     ((looking-at "heightfield")
      (setq pov-completion-list pov-heightfield-keywords))
     ((looking-at "prism")
      (setq pov-completion-list pov-prism-keywords))
     ((looking-at "bicubic")
      (setq pov-completion-list pov-bicubic-keywords))
     ((looking-at "light_source")
      (setq pov-completion-list pov-light-source-keywords))
     ((looking-at "interior")
      (setq pov-completion-list pov-interior-keywords))
     ((looking-at "media")
      (setq pov-completion-list pov-media-keywords))
     ((looking-at "fog")
      (setq pov-completion-list pov-fog-keywords))
     ((looking-at "global_settings")
      (setq pov-completion-list pov-global-settings-keywords))
     ((looking-at "radiosity")
      (setq pov-completion-list pov-radiosity-keywords))
     ((looking-at pov-csg-scope-re)
      (setq pov-completion-list (append pov-solid-primitive-keywords pov-infinite-solid-keywords pov-object-modifier-keywords pov-csg-keywords)))
     (t
      (setq pov-completion-list (append pov-object-modifier-keywords pov-object-keywords))))
     (setq pov-completion-list (append pov-completion-list pov-transformation-keywords)))
      (setq pov-completion-list (append pov-top-level-keywords pov-solid-primitive-keywords pov-infinite-solid-keywords pov-patch-primitive-keywords pov-csg-keywords)))))

(defun pov-completion (pov-completion-str pov-completion-pred 
                                          pov-completion-flag)
  (save-excursion
    (let ((pov-completion-all nil))
      (pov-get-scope)
      (mapcar '(lambda (s)
                 (if (string-match (concat "\\<" pov-completion-str) s)
                     (setq pov-completion-all (cons s pov-completion-all))))
              pov-completion-list)
      ;; Now we have built a list of all matches. Give response to caller
      (pov-completion-response))))

(defun pov-completion-response ()
  (cond ((or (equal pov-completion-flag 'lambda) (null pov-completion-flag))
         ;; This was not called by all-completions
         (if (null pov-completion-all)
             ;; Return nil if there was no matching label
             nil
           ;; Get longest string common in the labels
           (let* ((elm (cdr pov-completion-all))
                  (match (car pov-completion-all))
                  (min (length match))
                  tmp)
             (if (string= match pov-completion-str)
                 ;; Return t if first match was an exact match
                 (setq match t)
               (while (not (null elm))
                 ;; Find longest common string
                 (if (< (setq tmp (pov-string-diff match (car elm))) min)
                     (progn
                       (setq min tmp)
                       (setq match (substring match 0 min))))
                 ;; Terminate with match=t if this is an exact match
                 (if (string= (car elm) pov-completion-str)
                     (progn
                       (setq match t)
                       (setq elm nil))
                   (setq elm (cdr elm)))))
             ;; If this is a test just for exact match, return nil ot t
             (if (and (equal pov-completion-flag 'lambda) (not (equal match 't)))
                 nil
               match))))
        ;; If flag is t, this was called by all-completions. Return
        ;; list of all possible completions
        (pov-completion-flag
         pov-completion-all)))

(defun pov-complete-word ()
  "Complete word at current point based on POV syntax."
  (interactive)
  (let* ((b (save-excursion (skip-chars-backward "a-zA-Z0-9_") (point)))
         (e (save-excursion (skip-chars-forward "a-zA-Z0-9_") (point)))
         (pov-completion-str (buffer-substring b e))
         ;; The following variable is used in pov-completion
;        (pov-buffer-to-use (current-buffer))
         (allcomp (all-completions pov-completion-str 'pov-completion))
         (match (try-completion
                 pov-completion-str (mapcar '(lambda (elm)
                                        (cons elm 0)) allcomp))))

    ;; Delete old string
    (delete-region b e)

    ;; Insert match if found, or the original string if no match
    (if (or (null match) (equal match 't))
        (progn (insert "" pov-completion-str)
               (message "(No match)"))
      (insert "" match))
      ;; Give message about current status of completion
    (cond ((equal match 't)
           (if (not (null (cdr allcomp)))
               (message "(Complete but not unique)")
             (message "(Sole completion)")))
          ;; Display buffer if the current completion didn't help 
          ;; on completing the label.
          ((and (not (null (cdr allcomp))) (= (length pov-completion-str)
                                              (length match)))
           (with-output-to-temp-buffer "*Completions*"
             (display-completion-list allcomp))
             ;; Wait for a keypress. Then delete *Completion*  window
           (momentary-string-display "" (point))
           (delete-window (get-buffer-window (get-buffer "*Completions*")))))))


;;; initialize the keymap if it doesn't already exist
(if (null pov-mode-map)
    (progn
      (setq pov-mode-map (make-sparse-keymap))
      (define-key pov-mode-map "{" 'pov-open)
      (define-key pov-mode-map "}" 'pov-close)
      (define-key pov-mode-map "\t" 'pov-tab)
      (define-key pov-mode-map "\r" 'pov-newline)
      (define-key pov-mode-map "\C-c\C-c" 'pov-command-query) ;AS
      (define-key pov-mode-map [(shift f1)] 'pov-keyword-help) ;AS
      (define-key pov-mode-map "\C-c\C-l" 'pov-show-render-output) ;AS
      (define-key pov-mode-map "\M-\t" 'pov-complete-word)))

;; Hack to redindent end/else/break
(if pov-autoindent-endblocks
    (progn
      (define-key pov-mode-map "e" 'pov-autoindent-endblock)
      (define-key pov-mode-map "k" 'pov-autoindent-endblock)
      (define-key pov-mode-map "d" 'pov-autoindent-endblock)))

;;;;AS
;;; povkeyword help
(defun pov-keyword-help nil
"look up the appropriate place for keyword in the POV documentation"
"keyword can be entered and autocompleteted, default is word at point /AS"
  (interactive)
  (let* ((default (current-word))
	 (input (completing-read
		 (format "lookup keyword (default %s): " default)
		 pov-keyword-completion-alist))
	 (kw (if (equal input "")
		 default
	       input)))
    (get-buffer-create pov-doc-buffer-name)
    (switch-to-buffer-other-window pov-doc-buffer-name)
    (find-file-read-only (concat pov-home-dir pov-help-file))
    ;;Try to look up a keyword in the povray-documentation:
    ;;uses a heuristic to find the appropriate entry
    ;;since the povray-docu is formatted rather arbitrarily
    
    ;;try:
    (cond
     ((progn
       (goto-char (point-min))
       (search-forward-regexp
	;;first: the language description is in section four, so look for:
	(concat
	 "^4\\.[0-9]+\\(\\.[0-9]+\\)?\\(\\.[0-9]+\\)?\\(\\.[0-9]+\\)?[ 	]+"
	 ;;change light_source -> light_source OR light source (that's
	 ;;the usual spelling in the headings)
	 ;;(wouldn't a working replace-in-string be nice, even for
	 ;;FSF-emacs ???)
	 (if (string-match "\\(.*\\)_\\(.*\\)" kw)
	     (concat (match-string 1 kw) 
		     "[_ 	]"
		     (match-string 2 kw)
		     ".*\\>$")
	   kw))
	nil t))			;return nil if not found
       ;; make the line of the match the top line of screen XXX      
       (recenter 0))		
     
     ;;second: if that didn't work:
     ;;same again with a relaxed regexp that allows more matches:
     ((progn
	(goto-char (point-min))
	(search-forward-regexp
	 (concat
	  "^4\\.[0-9]+\\(\\.[0-9]+\\)?\\(\\.[0-9]+\\)?\\(\\.[0-9]+\\)?[ 	]+.*"
	  (if (string-match "\\(.*\\)_\\(.*\\)" kw)
	      (concat 
	       (match-string 1 kw) 
	       "[_ 	]"
	       (match-string 2 kw))
	    kw))
	 nil t))			
      (recenter 0))		
     ;;third try:
     ;;syntactic definitions appear like this: "KEYWORD:"
     ((progn
	(goto-char (point-min))
	(search-forward-regexp
	 (concat "\\<" (upcase kw) ":")
	 nil t)))
      ;;last try: simply search keyword from beginning of buffer
     ((progn
       (goto-char (point-min))
       (while (y-or-n-p (concat "Continue to look for " kw))
	 (search-forward-regexp
	  (concat  "\\<" kw  "\\>")
	  nil t))))
     ;;OK, that's it: we failed
     (t (error (concat "Couldn't find keyword: " kw
		      ", maybe you misspelled it?"))))
    ))

;;; Execution of Povray and View processes
(defvar pov-next-default-command "Render" ;XXX
  "The default command to run next time pov-command-query is run"
  )
(defvar pov-last-render-command "Render" ;XXX
  "The last command used to render a scene")
(defvar pov-rendered-succesfully nil
  "Whether the last rendering completed without errors")
(defvar pov-doc-buffer-name "*Povray Doc*"
  "The name of the buffer in which the documentation will be displayed")
;; will be set to *Pov Render <buffer-name>*
(defvar pov-render-buffer-name ""
  "The name of the buffer that contains the rendering output")
(defvar pov-current-render-process nil
  "The process rendering at the moment or nil")
(defvar pov-current-view-processes (make-hash-table)
  "The processes that display pictures at the moment")
(defvar pov-buffer-modified-tick-at-last-render 0
  "The number of modifications at time of last render")

;;make all the render variables buffer-local that are pov-file
;;dependent, so that users can render more than one file at the same
;;time etc.  Note: for the *view processes* a hash is used (rather
;;then making the variables local, because somebody might want to view
;;a file from a different render buffer.

(mapc 'make-variable-buffer-local

      '(pov-command-alist ;because of history XXX
	pov-next-default-command
	pov-last-render-command
	pov-image-file
	pov-render-buffer-name
	pov-buffer-modified-tick-at-last-render
	pov-current-render-process))
;DEBUG
;(set-default
;(defvar current-pov-file nil
;  "Name of the pov-file we are editing"
;  )
(defvar pov-image-file ""
  "The name of the rendered image that should be displayed"
  )

(defun pov-default-view-command ()
  "Return the default view command (internal or external)"
  (if pov-default-view-internal
      pov-internal-view
    pov-external-view))
  
(defun pov-command-query ();XXX
  "Query the user which command to execute"
  ;;XXX this one is still a mess
  (interactive)
  ;;Check whether the buffer has been modified since last call,
  ;;and the last rendering was succesful. If so he probably
  ;;wants to render, otherwise he wants to view.
  (let* ((default
	   (if (and (= (buffer-modified-tick)
		       pov-buffer-modified-tick-at-last-render)
		    pov-rendered-succesfully)
	       (pov-default-view-command)
	     pov-last-render-command))
	 (completion-ignore-case t)
	 (pov-command (completing-read
		       (format "Which command (default: %s)? " default)
		       pov-command-alist nil t nil t)))
    (setq pov-command
	  (if (not (string-equal pov-command ""))
	      pov-command
	    default))
    (setq pov-next-default-command pov-command)
  ;;XXX argl: all this information should be in pov-command-alist
  (cond ((string-match pov-command pov-internal-view)
	 (pov-display-image-xemacs pov-image-file)) ;XXX
	((string-match pov-command pov-external-view)
	 (pov-display-image-externally pov-image-file))
	(t
	   (setq pov-buffer-modified-tick-at-last-render
		 (buffer-modified-tick))
					;(message (format
					;	       "DEBUG: buffer %s modified tick%d "
					;	       (buffer-name)
					;	       (buffer-modified-tick)))
	   (pov-render-file pov-command (buffer-file-name))
	   ))))

(defun pov-render-file (pov-command file)
  "Render a file using pov-command."
  ;;XXX Check that there isn't already a render running
  (when
      (or
       (not pov-current-render-process)
       (and pov-current-render-process
	    (cond ((y-or-n-p
		    ;;XXX could theoretically be also running in other buffer...
		    "There is a render process already running: abort it?")
		   (kill-process pov-current-render-process)
		   (message "Process killed")
		   t)
		  )))
    (let ((render-command nil)
	  (render-command-options nil)
	  (render-command-history nil)
	  (old-buffer (current-buffer))
	  (process nil))
      
      ;; if the user hasn't saved his pov-file, ask him
    (if (buffer-modified-p) 
	(and (y-or-n-p
	      (concat (buffer-name (current-buffer)) " modified; save ? "))
	     (save-buffer)))
    ;; assign the buffer local value of the render buffer name
    (setq pov-render-buffer-name (format "*Povray Render %s*" file))
    (set-buffer (get-buffer-create pov-render-buffer-name))
    (erase-buffer)
    (setq render-command (second (assoc pov-command pov-command-alist)))
    (setq render-command-options (format
				  (third (assoc pov-command pov-command-alist))
				  file))
    (setq render-command-history
	  (fourth (assoc pov-command pov-command-alist)))
    ;(message (format "DEBUG FUCK %s %s"render-command-options (or
    ;		render-command-history "NIL")))
    (setq render-command-options
	  (read-string "Render with the following options: "
		       render-command-options
		       'render-command-history))
    (message (format "Running %s on %s" pov-command file))
    (insert (format "Running %s on %s with: %s %s..." pov-command file
		    render-command  render-command-options))

    (setq process (apply 'start-process pov-command (current-buffer)
			 render-command
			 (split-string render-command-options)))
    ;; memorize what we are doing
    (setq pov-last-render-command pov-command)
    ;; FIXME this might be dubious
    (setf (fourth (assoc pov-command pov-command-alist))
	  render-command-history)
    (message (format "DEBUG proc: %s" process))
    ;;XXX 'coz pov-current-render-process is buffer-local
    (get-buffer old-buffer)
    (setq pov-current-render-process process)
    (set-process-filter process 'pov-render-filter)
    (set-process-sentinel process 'pov-render-sentinel))))
  
(defun pov-show-render-output ()
  "Pop up the output of the last render command."
  (interactive)
  (let ((buffer (get-buffer pov-render-buffer-name)))
    (if buffer
	(let ((old-buffer (current-buffer)))
	  (pop-to-buffer buffer t)
	  (bury-buffer buffer)
	  (goto-char (point-max))
	  (pop-to-buffer old-buffer))
      (error "No rendering done so far"))))

(defun pov-render-sentinel (process event)
 "Sentinel for povray call."
 ;;so we aren't rendering any more ;XXX
 (setq pov-current-render-process nil)
 ;;If the process exists successfully then kill the ouput buffer
 (cond ((equal 0 (process-exit-status process))
	(setq pov-rendered-succesfully t)
	(message "Image rendered succesfully"))
       (t
	(message (concat "Errors in " (process-name process)
			", press C-c C-l to display"))
	(setq pov-rendered-succesfully nil))))

  
(defun pov-render-filter (process string)
  "Filter to process povray output. Scrolls and extracts the
filename of the output image (XXX with a horrible buffer-local-hack...)"
  (message (format "DEBUG buffer name %s" (buffer-name (current-buffer))))
  (let ((image-file nil))
    (save-excursion
      (set-buffer (process-buffer process))
      (save-excursion
	;; find out how our file is called
	(if (string-match "^ *Output file: \\(.*\\), [0-9]+ bpp .+$" string)
	    (setq image-file (match-string 1 string)))
	(goto-char (process-mark process))
	(insert-before-markers string)
	(set-marker (process-mark process) (point))))
    (if image-file  (setq pov-image-file image-file))))

(defun pov-display-image-externally (file)
  "Display the rendered image using external viewer"
  ;;if we don't have a file, prompt for one
  (when (or (not file) (string-equal file ""))
    (setq file
	  (read-file-name "Which image file should I display? ")))
  (let ((view-command nil)
	(view-options nil)
	(view-history nil)
	(other-view (cl-gethash file pov-current-view-processes))
	(process nil))
    (if (and other-view (processp other-view))  ;external
	(if (not (y-or-n-p
		  (format "Do yo want to want to kill the old view of %s?" file)))
	    (kill-process other-view)))
    (setq view-command (second (assoc pov-external-view pov-command-alist)))
    (setq view-options (format
			(third (assoc pov-external-view pov-command-alist))
			file))
    (setq view-history (fourth (assoc pov-external-view pov-command-alist)))
    (setq view-options (read-string "View with the following options: "
				    view-options
				    view-history))
    (message (format "Viewing %s with %s %s" file view-command view-options))
    (setq process (apply 'start-process (concat pov-external-view file) nil
			 view-command (split-string view-options)))
    ;;; remember what we have done
    (cl-puthash file process pov-current-view-processes)
    ;; update history
    (setf (fourth (assoc pov-external-view pov-command-alist)) view-history)
    ;;Sentinel for viewer call (XXX argl, what a hack)
    (set-process-sentinel
     process
     '(lambda (process event)
	;;seems like we finished viewing => remove process from hash
	(cl-remhash file pov-current-view-processes)
	(if (equal 0 (process-exit-status process))
	    (message (concat "view completed successfully")) ;XXX
	  (message (format "view exit status %d"
			   (process-exit-status process))))))))
  
(defun pov-display-image-xemacs (file)
  "Display the rendered image in a Xemacs frame"
  ;;TODO: set frame according to image-size (seems difficult)
  (when (or (not file) (string-equal file ""))
      (setq file
	    (read-file-name "Which image file should I display? ")))
  (let ((buffer (get-buffer-create
		 (format "*Povray View %s*" file))))
    (save-excursion
      (set-buffer buffer)
      (toggle-read-only -1)
      (erase-buffer)
      (insert-file-contents file)
      (toggle-read-only 1)
      ;;this will either bring the old frame with the picture to the forground
      ;;or create a new one
      (make-frame-visible 
       (or (get-frame-for-buffer (current-buffer))
	   (get-frame-for-buffer-make-new-frame (current-buffer)))))))
  ;(concat 
  ; (third (assoc pov-command pov-command-alist))
  ; file

;;;imenu stuff FIXME: doesn't seem to work at all
(defvar pov-imenu-generic-expression
  ;"^\\(camera[ 	]*\\)\\|\\(#declare\\)"
      '("^#declare[ \t]+\\([a-zA-Z0-9_]+\\)" . (1))
      )
;\(camera[ 	]*\)\|\(#declare\)
;(defun pov-imenu-extract-pov-index-name ()
;  (let ((index-alist '())
;	)
;    (save-excursion
;      (goto-char (point-min))
;      (save-match-data
;	(while (re-search-forward pov-menu-regexp nil t)
;	  (push (imenu-example--name-and-position) index-alist)))
;      (nreverse index-alist))))
	
(make-local-variable 'imenu-generic-expression)
(setq imenu-generic-expression pov-imenu-generic-expression)
	
(provide 'pov-mode)
;;; pov-mode.el ends here